#!/usr/bin/perl -w

# Program mftparser.pl

# (c) Gary C. Kessler, 2012-2013 [gck@garykessler.net]

# Program to parse an NTFS Master File Table (MFT) (FILE) entry
# Ref: File System Forensic Analysis, Brian Carrier,
# http://msdn.microsoft.com/en-us/library/bb470206%28VS.85%29.aspx,
# dubeyko.com/development/FileSystems/NTFS/ntfsdoc.pdf,
# http://www.reddragonfly.org/ntfs/concepts/file_record.html,
# http://www.reddragonfly.org/ntfs/concepts/data_runs.html,
# http://inform.pucp.edu.pe/~inf232/Ntfs/ntfs_doc_v0.5/concepts/attribute_header.html,
# http://www.ntfs.com/ntfs_basics.htm, and other documentation

# NTFS to epoch time conversion:
# http://support.citrix.com/article/CTX109645

# Thanks to Gregg Gunsch, Defiance College, for ever-patient debugging assistance!
# Any remaining errors are my own!!!

# General format -- An MFT entry is 1024 B in length. The Header is nominally 42 B
# (usually longer) followed by the Content, which is composed of Attributes. Attributes,
# in turn, comprise a Header and Content, 22 B and variable-length, 
# respectively for a resident attribute, and >64 B and 0 B, respectively,
# for a non-resident attribute. The remainder of the MFT entry is fill.

# ********************************************************
# MAIN PROGRAM BLOCK
# ********************************************************

#use Encode qw/encode_utf8 decode_utf8/;
use utf8;
binmode STDOUT, ":utf8";

# Initialize global variables

$version = "2.1";
$build_date = "11 June 2013";

# *** This program continues to be an iterative, work-in-progress. ***

# Initialize variables...

# $source is the name of the file to read.
# $first and $last are the range of MFT records to display. Default to
# all (i.e., $first = 0, $last = -1) [Thanks, Gregg!]

$source = "";
$verbose = 0;
$first = 0; $last = -1;

#
# Begin program activity!!
#

print "\nMaster File Table (MFT) Parser V$version - Gary C. Kessler ($build_date)\n\n";

if (parsecommandline ())
  { exit 1; };

if (not (-e $source))
  {
  print "\nSource file $source does not exist\n\n";
  exit 2;
  }

$filesize = -s ($source);
$max_mfts = int ($filesize / 1024) - 1;
print "Source file $source ($filesize bytes) contains ", $max_mfts+1, " MFT record(s)\n";

if ($first > $max_mfts || $first < 0)
  {
  print "Invalid starting MFT record number $first\n\n";
  exit 3;
  }
if ($last < $first || $last > $max_mfts) { $last = $max_mfts; }
if ($first > 0 || $last < $max_mfts)
  { print "Displaying MFT records $first-$last\n"; }

print "\n";

# Open the file, then read/process one MFT record at a time

open (INFILE,"<",$source);
binmode (INFILE);

# Skip directly to first record if not starting at 0.

if ($first != 0)
  { seek (INFILE, $first*1024, 0); }

for ($mft_rec=$first; $mft_rec<=$last; $mft_rec++)
  {

# Read one entire MFT

  for ($i=0; $i<1024; $i++)
    {
    read (INFILE,$byte,1);
    $mft [$i] = unpack ("C", $byte);
    }

  print "==================== MFT RECORD #$mft_rec ====================\n";

  if ($verbose == 1)
    { 
    print "RAW DATA\n";
    list_bytes (0,1024,@mft);
    print "\n";
    }

# Parse the MFT Header

  parse_MFT_header ();

# Parse the MFT Contents (i.e., attribute headers and contents) 

  parse_MFT_contents ();

  print "\n";
  }

close (INFILE);


# ********************************************************
# commify
# ********************************************************

# Routine from PerlFAQ5 -- add commas to large integers
# http://perldoc.perl.org/perlfaq5.html
  #How-can-I-output-my-numbers-with-commas-added?

sub commify
{
local $_  = shift;
1 while s/^([-+]?\d+)(\d{3})/$1,$2/;
return $_;
}

# ********************************************************
# flag_handler
# ********************************************************

# Print out flag values from bitmap interpreted from a table
# $val is the flag value, $n is the byte size of $val, and
# %table is the flag table

sub flag_handler ($$@)
{
my ($val,$n,%table) = @_;
my ($a, $b, $flag, $temp);

$temp = $val;
if ($temp == 0x0000)
    { print "(None)\n"; }
  else
    {
    $a = 0;
    for ($b=0; $b<8*$n; $b++)
      {
      if ($temp%2 == 1)
        {
        if ($a > 0)
          { print ", "; }
        $flag = 2**$b;
        if (defined $table{$flag})
            { print "$table{$flag}"; }
          else
            { print "(Not listed)"; }
        $a++;
        }
      $temp = int ($temp / 2);
      }
    print "\n";
    }
}

# ********************************************************
# help_text
# ********************************************************

# Display the help file

sub help_text
{
print<< "EOT";
Program usage: mftparser [-i input_file] [-v] [-f first_MFT] [-l last_MFT]
               mftparser {[-h] | [-t]}

 where: -i is the input file name
        -f is the first MFT_record to display (counting starts at 0)
        -l is the last MFT_record to display (-1 or blank "to the end")
        -h prints this help file
        -t displays the MFT template
        -v (verbose) displays the entire MFT record in hex

If no -i switch is provided, the program will ask for the
input file name.

The input file should contain the contents of one or more 1,024-byte
FILE records in a binary file. The best way to obtain a FILE record
is to export the file from your forensics software or a hex editor.

EOT
return;
}

# ********************************************************
# hex_to_signed_integer
# ********************************************************

# Routine takes a hex string ($hexstr) that is $n bytes in size
# and converts to signed integer. This is a way to derive a
# positive or negative number from a hex string.

sub hex_to_signed_integer
{
my ($hexstr,$n) = @_;
my $num;

print "\nInvalid hex string: $hexstr\n"
  if $hexstr !~ /^[0-9A-Fa-f]{1,8}$/;
 
$num = hex($hexstr);
return $num >> $n*8-1 ? $num - 2 ** ($n*8) : $num;
}

# ********************************************************
# list_bytes
# ********************************************************

# Given the array and a number of bytes, set the output to 
# dump X 16-byte lines followed by a single line with the
# rest of the bytes

sub list_bytes
{
my ($start,$n,@array) = @_;
my ($j, $k, $full_lines, $short_line);

# Determine how many full lines to print, assuming 16 B/line

$full_lines = int ($n/16);
$short_line = $n - ($full_lines * 16);

$k = $start;

for ($j=0; $j<$full_lines; $j++)
  {
  set_output ($k,16,@array);
  $k += 16;
  }

if ($short_line > 0)
  { set_output ($k,$short_line,@array); }

print "\n";
}

# ********************************************************
# list_hex_digits
# ********************************************************

# Given an array name, starting position, and number of digits, print
# out a string of hex digits in the format 0xAA-BB-CC...

sub list_hex_digits ($$@)
{
my ($start,$n,@array) = @_;
my $i;

for ($i = $start; $i < $start+$n; $i++)
  {
  if ($i == $start)
      { print "0x"; }
    else
      { print "-"; }
  printf ("%02X", $array[$i]);
  }

return;
}

# ********************************************************
# little_endian
# ********************************************************

# Given an array name, starting position, and number of digits,
# calculate the decimal value using little endian (LSB first)

sub little_endian ($$@)
{
my ($start,$n,@array) = @_;
my ($i, $sum);

$sum = $array[$start];
for ($i=1; $i<$n; $i++)
  { $sum += $array[$start+$i] * (2**($i*8)); }

return $sum;
}

# ********************************************************
# parsecommandline
# ********************************************************

# Parse command line for file name, if present.

# Return $state = 1 to indicate that the program should stop
# immediately (switch -h)

sub parsecommandline
{
my $state = 0;

# Parse command line switches ($ARGV array of length $#ARGV)

if ($#ARGV >= 0)
  { 
  for ($i = 0; $i <= $#ARGV; $i++)

    {
    PARMS:
      {
      $ARGV[$i] eq "-i" && do
         {
         $source = $ARGV[$i+1];
         $i++;
         last PARMS;
         };
      $ARGV[$i] eq "-f" && do
         {
         $first = $ARGV[$i+1];
         $i++;
         last PARMS;
         };
      $ARGV[$i] eq "-l" && do
         {
         $last = $ARGV[$i+1];
         $i++;
         last PARMS;
         };
      $ARGV[$i] eq "-v" && do
         {
         $verbose = 1;
         last PARMS;
         };
      $ARGV[$i] eq "-t" && do
         {
         $layout = "layout_MFT.txt";
         open (INFILE, "<", $layout)
          or die "Can't open input file \"$layout\": $!";
         while (not (eof INFILE))
           {
           $line = <INFILE>;
           print $line;
           }
         close INFILE;
         $state = 1;
         return $state;
         };
      $ARGV[$i] eq "-h" && do
         {
         help_text();
         $state = 1;
         return $state;
         };

      do
         {
         print "Invalid parameter \"$ARGV[$i]\".\n\n";
         print "Usage: mftparser [-i input_file] [-v] [-f first_MFT] [-l last_MFT]\n";
         print "       mftparser {[-h] | [-t]}\n\n";
         $state = 1;
         return $state;
         };
      };
    };
  };

# Prompt for file name, if not supplied

if ($source eq "")
  {
  print "Enter the input file name: ";
  chomp ($source = <STDIN>);
  print "\n";
  };

return $state;
}

# ********************************************************
# parse_MFT_contents
# ********************************************************

# Parse the MFT contents (i.e., attribute headers and contents)

sub parse_MFT_contents
{
my ($attrib_len, $attrib_name, $attrib_num, $attrib_start, $attrib_type, $cluster_num);
my ($content_len, $content_offset, $cus, $fn_len, $j, $k, $name_len, $name_offset);
my ($num_clus, $relative, $resident_flag, $run_length, $run_length_len, $run_offset);
my ($run_offset_len, $runlist_end, $runlist_offset, $runlist_start, $temp);

# Set up the tables...

# MFT Attribute Type

my %attribute_type =
  (
   16 => "\$STANDARD_INFORMATION",
   32 => "\$ATTRIBUTE_LIST",
   48 => "\$FILE_NAME",
   64 => "\$VOLUME_VERSION/\$OBJECT_ID",
   80 => "\$SECURITY_DESCRIPTOR",
   96 => "\$VOLUME_NAME",
  112 => "\$VOLUME_INFORMATION",
  128 => "\$DATA",
  144 => "\$INDEX_ROOT",
  160 => "\$INDEX_ALLOCATION",
  176 => "\$BITMAP",
  192 => "\$SYMBOLIC_LINK/\$REPARSE_POINT",
  208 => "\$EA_INFORMATION",
  224 => "\$EA",
  256 => "\$LOGGED_UTILITY_STREAM"
  );

# Attribute Header flags

my %AH_flags =
  (
  0x0001 => "Compressed",
  0x4000 => "Encrypted",
  0x8000 => "Sparse"
  );

# $STANDARD_INFORMATION (16) and $FILE_NAME (48) Flags

my %SI_flag_values =
  (
  0x00000001 => "Read-only",
  0x00000002 => "Hidden file",
  0x00000004 => "System file",
  0x00000020 => "Archive",
  0x00000040 => "Device",
  0x00000080 => "Normal",
  0x00000100 => "Temporary",
  0x00000200 => "Sparse file",
  0x00000400 => "Reparse point",
  0x00000800 => "Compressed",
  0x00001000 => "Offline",
  0x00002000 => "Content not being indexed for faster searches",
  0x00004000 => "Encrypted",
  0x10000000 => "Directory",
  0x20000000 => "Index view"
  );

# $FILE_NAME (48) Namespace

my %FN_namespace =
  (
  0 => "POSIX",
  1 => "Win32",
  2 => "DOS",
  3 => "Win32/DOS"
  );

# The real action starts here!!!!

print "\n********** MFT CONTENTS **********\n";

$attrib_num = 0;

# Get attribute type. Cycle through until type = 0xFFFFFFFF

while (0 == 0)
  {

# Bytes 0-4 are the attribute type.

  $attrib_start = $bpos;
  $len = 4;
  $attrib_type = little_endian ($bpos,$len,@mft);

# Break out of the while loop when 0xFFFFFFFF is found

  if ($attrib_type == 0xFFFFFFFF)
    {
    printf ("%04d-%04d  Attribute type: ", $bpos, $bpos+$len-1);
    list_hex_digits ($bpos,$len,@mft);
    print " [END_OF_ATTRIBUTE_LIST]\n";
    $bpos += $len;
    last;
    }

  $attrib_num++;
  $attrib_name = $attribute_type {$attrib_type};
  print "\nATTRIBUTE #$attrib_num ($attrib_name)\n";
  print "-------------------- HEADER --------------------\n";
  printf ("%04d-%04d  Attribute type: ", $bpos, $bpos+$len-1);
  list_hex_digits ($bpos,$len,@mft);
  print " [", $attrib_name, "]\n";

# Bytes 5-7 are the attribute length (including header + content)

  $bpos += $len; $len = 4;

# Ok, here's a bit of a kludge. The Attribute_Length field is,
# according to all documentation, four bytes in length so
# should use this code:

#  $attrib_len = little_endian ($bpos,$len,@mft);

# As far as I can tell, the two high-order bytes are ignored... so, I
# ignore them as follows:

  $attrib_len = little_endian ($bpos,2,@mft);
  printf ("%04d-%04d  Attribute length: ", $bpos, $bpos+$len-1);
  list_hex_digits ($bpos,$len,@mft);
  print " [", commify ($attrib_len), " bytes";
  if (little_endian ($bpos,$len,@mft) >= 2**16)
    { print " (ignoring two high-order bytes)"; }
  print "]\n";

# Byte 8 is the resident attribute flag

  $bpos += $len; $len = 1;
  $resident_flag = little_endian ($bpos,$len,@mft);
  printf ("%04d       Non-resident flag: ", $bpos);
  list_hex_digits ($bpos,$len,@mft);
  if ($resident_flag == 0)
      { print " [Attribute is resident]\n"; }
    else
      { print " [Attribute is non-resident]\n"; }

# Byte 9 is the attribute name length (Unicode characters)

  $bpos += $len; $len = 1;
  $name_len = little_endian ($bpos,$len,@mft);
  printf ("%04d       Name length: ", $bpos);
  list_hex_digits ($bpos,$len,@mft);
  print " [", commify ($name_len), " 16-bit characters]\n";

# Bytes 10-11 are the offset to the name

  $bpos += $len; $len = 2;
  $name_offset = little_endian ($bpos,$len,@mft);
  printf ("%04d-%04d  Offset to name: ", $bpos, $bpos+$len-1);
  list_hex_digits ($bpos,$len,@mft);
  print " [", commify ($name_offset), " bytes]\n";

# Bytes 12-13 are the flag bits (see table)

  $bpos += $len; $len = 2;
  $temp = little_endian ($bpos,$len,@mft);
  printf ("%04d-%04d  Flag: ", $bpos, $bpos+$len-1);
  list_hex_digits ($bpos,$len,@mft);
  print " -- ";
  flag_handler ($temp,$len,%AH_flags);

# Bytes 14-15 are the attribute id.

  $bpos += $len; $len = 2;
  $temp = little_endian ($bpos,$len,@mft);
  printf ("%04d-%04d  Attribute identifier: ", $bpos, $bpos+$len-1);
  list_hex_digits ($bpos,$len,@mft);
  print " [", commify ($temp), "]\n";

# Remaining bytes depend upon whether this is a resident or non-resident attribute

  if ($resident_flag == 0)
      {

# *** RESIDENT ATTRIBUTE ***

    # Bytes 16-19 are the content length

      $bpos += $len; $len = 4;
      $content_len = little_endian ($bpos,$len,@mft);
      printf ("%04d-%04d  Content length: ", $bpos, $bpos+$len-1);
      list_hex_digits ($bpos,$len,@mft);
      print " [", commify ($content_len), " bytes]\n";

    # Bytes 20-21 are the content offset

      $bpos += $len; $len = 2;
      $content_offset = little_endian ($bpos,$len,@mft);
      printf ("%04d-%04d  Content offset: ", $bpos, $bpos+$len-1);
      list_hex_digits ($bpos,$len,@mft);
      print " [", commify ($content_offset), "]\n";

    # The next fields are:
    #  1) Filler bytes, if needed for 32-bit alignment
    #  2) Attribute name, if present
    #  3) Attribute content

    # Start by checking for fill bytes based either on name offset or
    # content offset

      $bpos += $len;

    # Determine if there are "filler" bytes.

      if ($name_len != 0)
          { $len = $attrib_start + $name_offset - $bpos; }
        else
          { $len = $attrib_start + $content_offset - $bpos; }

      if ($len != 0)
        {
        printf ("%04d-%04d  (Fill): ", $bpos, $bpos+$len-1);
        list_hex_digits ($bpos,$len,@mft);
        print "\n";
        }

    # Now, print attribute name, if present.

      if ($name_len != 0)
        {
        $bpos += $len;
        $len = $name_len * 2;
        printf ("%04d-%04d  Name: ", $bpos, $bpos+$len-1);
        list_hex_digits ($bpos,$len,@mft);
        print " [";
        for ($k=0; $k<$name_len; $k++)
          {
          $temp = chr (little_endian ($bpos+$k*2,2,@mft));
          print $temp;
          }
        print "]\n";
        }

    # At this point, we're ready for the contents, which are only present in resident attributes

      if ($bpos+$len < $attrib_start + $content_offset)
        {
        $bpos += $len;
        $len = $attrib_start + $content_offset - $bpos;
        printf ("%04d-%04d  (Fill): ", $bpos, $bpos+$len-1);
        list_hex_digits ($bpos,$len,@mft);
        print "\n";
        }

      print "-------------------- CONTENTS --------------------\n";

# Parse the contents based upon the attrib_type. In some cases, just dump the hex content,
# in other cases, actually parse it.

# NOTE -- Some attributes do *not* have the full cimplement of information per the
# documentation. The next attribute should start at ($attrib_start+$attrib_len). If
# $bpos ever gets to that value, skip the rest of this attribute and assume that we
# have stumbled upon the next one.

      CONTENT:
        {
# Attribute 16 ($STANDARD_INFORMATION)

        $attrib_type == 16 && do
          {

    # Bytes 0-7 are the C-time

          $bpos += $len; $len = 8;
          $temp = little_endian ($bpos,$len,@mft);
          $temp = $temp /10000000 - 11644473600;
          printf ("%04d-%04d  C-time: ", $bpos, $bpos+$len-1);
          list_hex_digits ($bpos,$len,@mft);
          print " [", scalar (gmtime ($temp))," GMT]\n";

    # Bytes 8-15 are the M-time

          $bpos += $len; $len = 8;
          $temp = little_endian ($bpos,$len,@mft);
          $temp = $temp /10000000 - 11644473600;
          printf ("%04d-%04d  M-time: ", $bpos, $bpos+$len-1);
          list_hex_digits ($bpos,$len,@mft);
          print " [", scalar (gmtime ($temp))," GMT]\n";

    # Bytes 16-23 are the MFT modification (E)-time

          $bpos += $len; $len = 8;
          $temp = little_endian ($bpos,$len,@mft);
          $temp = $temp /10000000 - 11644473600;
          printf ("%04d-%04d  E-time: ", $bpos, $bpos+$len-1);
          list_hex_digits ($bpos,$len,@mft);
          print " [", scalar (gmtime ($temp))," GMT]\n";

    # Bytes 24-31 are the A-time

          $bpos += $len; $len = 8;
          $temp = little_endian ($bpos,$len,@mft);
          $temp = $temp /10000000 - 11644473600;
          printf ("%04d-%04d  A-time: ", $bpos, $bpos+$len-1);
          list_hex_digits ($bpos,$len,@mft);
          print " [", scalar (gmtime ($temp))," GMT]\n";

    # Bytes 32-35 are the Flags

          $bpos += $len; $len = 4;
          $temp = little_endian ($bpos,$len,@mft);
          printf ("%04d-%04d  Flags: ", $bpos, $bpos+$len-1);
          list_hex_digits ($bpos,$len,@mft);
          print " -- ";
          flag_handler ($temp,$len,%SI_flag_values);

    # Bytes 36-39 are the maximum number of versions

          if ($bpos+$len >= $attrib_start+$attrib_len) { last CONTENT; }
          $bpos += $len; $len = 4;
          $temp = little_endian ($bpos,$len,@mft);
          printf ("%04d-%04d  Max. number of versions: ", $bpos, $bpos+$len-1);
          list_hex_digits ($bpos,$len,@mft);
          print " [", commify ($temp), "]\n";

    # Bytes 40-43 are the version number

          if ($bpos+$len >= $attrib_start+$attrib_len) { last CONTENT; }
          $bpos += $len; $len = 4;
          $temp = little_endian ($bpos,$len,@mft);
          printf ("%04d-%04d  Version number: ", $bpos, $bpos+$len-1);
          list_hex_digits ($bpos,$len,@mft);
          print " [", commify ($temp), "]\n";

    # Bytes 44-47 are the class identifier

          if ($bpos+$len >= $attrib_start+$attrib_len) { last CONTENT; }
          $bpos += $len; $len = 4;
          $temp = little_endian ($bpos,$len,@mft);
          printf ("%04d-%04d  Class identifier: ", $bpos, $bpos+$len-1);
          list_hex_digits ($bpos,$len,@mft);
          print " [", commify ($temp), "]\n";

    # Bytes 48-51 are the owner identifier

          if ($bpos+$len >= $attrib_start+$attrib_len) { last CONTENT; }
          $bpos += $len; $len = 4;
          $temp = little_endian ($bpos,$len,@mft);
          printf ("%04d-%04d  Owner identifier: ", $bpos, $bpos+$len-1);
          list_hex_digits ($bpos,$len,@mft);
          print " [", commify ($temp), "]\n";

    # Bytes 52-55 are the security identifier

          if ($bpos+$len >= $attrib_start+$attrib_len) { last CONTENT; }
          $bpos += $len; $len = 4;
          $temp = little_endian ($bpos,$len,@mft);
          printf ("%04d-%04d  Security identifier: ", $bpos, $bpos+$len-1);
          list_hex_digits ($bpos,$len,@mft);
          print " [", commify ($temp), "]\n";

    # Bytes 56-63 are the quota charged

          if ($bpos+$len >= $attrib_start+$attrib_len) { last CONTENT; }
          $bpos += $len; $len = 8;
          $temp = little_endian ($bpos,$len,@mft);
          printf ("%04d-%04d  Quota charged: ", $bpos, $bpos+$len-1);
          list_hex_digits ($bpos,$len,@mft);
          print " [", commify ($temp), "]\n";

    # Bytes 64-71 are the update sequence number

          if ($bpos+$len >= $attrib_start+$attrib_len) { last CONTENT; }
          $bpos += $len; $len = 8;
          $temp = little_endian ($bpos,$len,@mft);
          printf ("%04d-%04d  Update sequence number: ", $bpos, $bpos+$len-1);
          list_hex_digits ($bpos,$len,@mft);
          print " [", commify ($temp), "]\n";

          last CONTENT;
          } ;

# Attribute 32 ($ATTRIBUTE_LIST)

        $attrib_type == 32 && do
          {

# 0-3, Attribute type
# 4-5, Length of this entry
# 6, Length of name (N)
# 7, Offset to name
# 8-15, Starting cluster number of name
# 16-23, File reference where attribute is located
# 24-25, Attribute identifier
# 26-(26+2N) Name in Unicode, if N>0

          last CONTENT;
          } ;

# Attribute 48 (FILE_NAME)

        $attrib_type == 48 && do
          {

    # Bytes 0-5 are the file reference of parent directory

          $bpos += $len; $len = 6;
          $temp = little_endian ($bpos,$len,@mft);
          printf ("%04d-%04d  Parent directory MFT #: ", $bpos, $bpos+$len-1);
          list_hex_digits ($bpos,$len,@mft);
          print " [", commify ($temp), "]\n";

    # Bytes 6-7 are the file reference of parent directory

          $bpos += $len; $len = 2;
          $temp = little_endian ($bpos,$len,@mft);
          printf ("%04d-%04d  Parent use/Delete count: ", $bpos, $bpos+$len-1);
          list_hex_digits ($bpos,$len,@mft);
          print " [", commify ($temp), "]\n";

    # Bytes 8-15 are the C-time

          $bpos += $len; $len = 8;
          $temp = little_endian ($bpos,$len,@mft);
          $temp = $temp /10000000 - 11644473600;
          printf ("%04d-%04d  C-time: ", $bpos, $bpos+$len-1);
          list_hex_digits ($bpos,$len,@mft);
          print " [", scalar (gmtime ($temp))," GMT]\n";

    # Bytes 16-23 are the M-time

          $bpos += $len; $len = 8;
          $temp = little_endian ($bpos,$len,@mft);
          $temp = $temp /10000000 - 11644473600;
          printf ("%04d-%04d  M-time: ", $bpos, $bpos+$len-1);
          list_hex_digits ($bpos,$len,@mft);
          print " [", scalar (gmtime ($temp))," GMT]\n";

    # Bytes 24-31 are the MFT modification (E)-time

          $bpos += $len; $len = 8;
          $temp = little_endian ($bpos,$len,@mft);
          $temp = $temp /10000000 - 11644473600;
          printf ("%04d-%04d  E-time: ", $bpos, $bpos+$len-1);
          list_hex_digits ($bpos,$len,@mft);
          print " [", scalar (gmtime ($temp))," GMT]\n";

    # Bytes 32-39 are the A-time

          $bpos += $len; $len = 8;
          $temp = little_endian ($bpos,$len,@mft);
          $temp = $temp /10000000 - 11644473600;
          printf ("%04d-%04d  A-time: ", $bpos, $bpos+$len-1);
          list_hex_digits ($bpos,$len,@mft);
          print " [", scalar (gmtime ($temp))," GMT]\n";

    # Bytes 40-47 are the allocated size of file

          $bpos += $len; $len = 8;
          $temp = little_endian ($bpos,$len,@mft);
          printf ("%04d-%04d  Allocated file size: ", $bpos, $bpos+$len-1);
          list_hex_digits ($bpos,$len,@mft);
          print " [", commify ($temp), " bytes]\n";

    # Bytes 48-55 are the actual file size

          $bpos += $len; $len = 8;
          $temp = little_endian ($bpos,$len,@mft);
          printf ("%04d-%04d  Actual file size: ", $bpos, $bpos+$len-1);
          list_hex_digits ($bpos,$len,@mft);
          print " [", commify ($temp), " bytes]\n";

    # Bytes 56-59 are the Flags

          $bpos += $len; $len = 4;
          $temp = little_endian ($bpos,$len,@mft);
          printf ("%04d-%04d  Flags: ", $bpos, $bpos+$len-1);
          list_hex_digits ($bpos,$len,@mft);
          print " -- ";
          flag_handler ($temp,$len,%SI_flag_values);

    # Bytes 60-63 are the reparse value

          $bpos += $len; $len = 4;
          $temp = little_endian ($bpos,$len,@mft);
          printf ("%04d-%04d  Reparse value: ", $bpos, $bpos+$len-1);
          list_hex_digits ($bpos,$len,@mft);
          print " [", commify ($temp), "]\n";

    # Byte 64 is the length of name

          $bpos += $len; $len = 1;
          $fn_len = little_endian ($bpos,$len,@mft);
          printf ("%04d       Length of file name: ", $bpos);
          list_hex_digits ($bpos,$len,@mft);
          print " [", commify ($fn_len), " characters]\n";

    # Byte 65 is the namespace

          $bpos += $len; $len = 1;
          $temp = little_endian ($bpos,$len,@mft);
          printf ("%04d       Namespace: ", $bpos);
          list_hex_digits ($bpos,$len,@mft);
          print " [", commify ($temp), "] (";
          if ($temp >= 1 && $temp <= 3)
              { print $FN_namespace{$temp}, ")\n"; }
            else
              { print "undefined)\n"; }

    # Bytes 66 et seq. are the file name

          $bpos += $len; $len = $fn_len * 2;
          printf ("       -------- File name --------\n");
          list_bytes ($bpos,$len,@mft);
          print "       ";
          for ($k=0; $k<$fn_len; $k++)
            {
            print chr (little_endian ($bpos+$k*2,2,@mft));
            }
          print "\n";

          last CONTENT;
          } ;

# Attribute 64 ($VOLUME_VERSION/$OBJECT_ID)

        $attrib_type == 64 && do
          {

    # Bytes 0-15 are the GUID Object Identifier

          $bpos += $len; $len = 16;
          $temp = little_endian ($bpos,$len,@mft);
          printf ("%04d-%04d  GUID Object Identifier: ", $bpos, $bpos+$len-1);
          list_hex_digits ($bpos,$len,@mft);
          print "\n";

          last CONTENT;
          } ;

# Attribute 80 ($SECURITY_DESCRIPTOR)

        $attrib_type == 80 && do
          {
          last CONTENT;
          } ;

# Attribute 96 ($VOLUME_NAME)

        $attrib_type == 96 && do
          {
          last CONTENT;
          } ;

# Attribute 112 ($VOLUME_INFORMATION)

        $attrib_type == 112 && do
          {

# Unicode name

          last CONTENT;
          } ;

# Attribute 128 ($DATA)

        $attrib_type == 128 && do
          {
          last CONTENT;
          } ;

# Attribute 144 ($INDEX_ROOT)

        $attrib_type == 144 && do
          {

    # Per Carrier
    # 0-3, Type of attribute in index (0 if entry does not use an attribute)
    # 4-7, Collation sorting rule
    # 8-11, Size of each index record (in bytes)
    # 12, Size of each index record (in clusters)
    # 13-15, Unused
    # 16+, Index node header

    # Index node header:
    # 0-3, Offset to start of index entry list (relative to start of the node header)
    # 4-7, Offset to end of used portion of index entry list (relative to start of the node header)
    # 8-11, Offset to end of allocated index entry list buffer (relative to start of the node header)
    # 12-15, Flags

    # Flags:
    # 0x00, Small Index (fits in Index Root)
    # 0x01, Large index (Index Allocation needed)

          last CONTENT;
          } ;

# Attribute 160 ($INDEX_ALLOCATION)

        $attrib_type == 160 && do
          {

    # Per Carrier
    # 0-3, Signature value ("INDX")
    # 4-5, Offset to fixup array
    # 6-7, Number of entries in fixup array
    # 8-15, $LogFile Sequence Number (LSN)
    # 16-23, The VCN of this record in the full index stream
    # 24+, Index node header

    # Index node header:
    # 0-3, Offset to start of index entry list (relative to start of the node header)
    # 4-7, Offset to end of used portion of index entry list (relative to start of the node header)
    # 8-11, Offset to end of allocated index entry list buffer (relative to start of the node header)
    # 12-15, Flags

    # Flags:
    # 0x01, Index entry points to a sub-node
    # 0x02, Last index entry in the node

          last CONTENT;
          } ;

# Attribute 176 ($BITMAP)

        $attrib_type == 176 && do
          {
          last CONTENT;
          } ;

# Attribute 192 ($SYMBOLIC_LINK/$REPARSE_POINT)

        $attrib_type == 192 && do
          {
          last CONTENT;
          } ;

# Attribute 208 ($EA_INFORMATION)

        $attrib_type == 208 && do
          {

    # 0x00 	2 	Size of the packed Extended Attributes
    # 0x02 	2 	Number of Extended Attributes which have NEED_EA set
    # 0x04 	4 	Size of the unpacked Extended Attributes
          last CONTENT;
          } ;

# Attribute 224 ($EA)

        $attrib_type == 224 && do
          {

    # 0x00 	4 	Offset to next Extended Attribute
    # 0x04 	1 	Flags
    # 0x05 	1 	Name Length (N)
    # 0x06 	2 	Value Length (V)
    # 0x08 	N 	Name
    # N+0x08 	V 	Value

    # Flags: 0x80 	Need EA

          last CONTENT;
          } ;

# Attribute 256 ($LOGGED_UTILITY_STREAM)

        $attrib_type == 256 && do
          {
          last CONTENT;
          } ;

        }

# Check to see if there are any bytes left in the Contents...

      $bpos += $len;

      if ($bpos < $attrib_len + $attrib_start)
        {
        if ($bpos != $attrib_start + $content_offset)
          { print "---- Remaining bytes in attribute contents ----\n"; }
        list_bytes ($bpos,$attrib_len+$attrib_start-$bpos,@mft);
        $bpos = $attrib_len + $attrib_start;
        }

      }
    else
      {

# *** NON-RESIDENT ATTRIBUTE ***

# Bytes 16-23 are the starting VCN number for the runlist

      $bpos += $len; $len = 8;
      $runlist_start = little_endian ($bpos,$len,@mft);
      printf ("%04d-%04d  Runlist starting VCN: ", $bpos, $bpos+$len-1);
      list_hex_digits ($bpos,$len,@mft);
      print " [", commify ($runlist_start), "]\n";

# Bytes 24-31 are the ending VCN number for the runlist

      $bpos += $len; $len = 8;
      $runlist_end = little_endian ($bpos,$len,@mft);
      printf ("%04d-%04d  Runlist ending VCN: ", $bpos, $bpos+$len-1);
      list_hex_digits ($bpos,$len,@mft);
      print " [", commify ($runlist_end), "]\n";

# Bytes 32-33 are the offset to the runlist

      $bpos += $len; $len = 2;
      $runlist_offset = little_endian ($bpos,$len,@mft);
      printf ("%04d-%04d  Offset to runlist: ", $bpos, $bpos+$len-1);
      list_hex_digits ($bpos,$len,@mft);
      print " [", commify ($runlist_offset), " bytes]\n";

# Bytes 34-35 are the compression unit size. 0 = uncompressed, otherwise
# unit size = 2**x clusters

      $bpos += $len; $len = 2;
      $cus = little_endian ($bpos,$len,@mft);
      printf ("%04d-%04d  Compression unit size: ", $bpos, $bpos+$len-1);
      list_hex_digits ($bpos,$len,@mft);
      if ($cus == 0)
          { print " [no compression]\n"; }
        else
          { print " [", commify (2**$temp), " clusters]\n"; }

# Bytes 36-39 are unused

      $bpos += $len; $len = 4;
      printf ("%04d-%04d  Unused: ", $bpos, $bpos+$len-1);
      list_hex_digits ($bpos,$len,@mft);
      print "\n";

# Bytes 40-47 are the allocated size of the attribute contents

      $bpos += $len; $len = 8;
      $temp = little_endian ($bpos,$len,@mft);
      printf ("%04d-%04d  Allocated attribute content size: ", $bpos, $bpos+$len-1);
      list_hex_digits ($bpos,$len,@mft);
      print " [", commify ($temp), " bytes]\n";

# Bytes 48-55 are the actual size of the attribute contents

      $bpos += $len; $len = 8;
      $temp = little_endian ($bpos,$len,@mft);
      printf ("%04d-%04d  Actual attribute content size: ", $bpos, $bpos+$len-1);
      list_hex_digits ($bpos,$len,@mft);
      print " [", commify ($temp), " bytes]\n";

# Bytes 56-63 are the initialized size of the attribute contents

      $bpos += $len; $len = 8;
      $temp = little_endian ($bpos,$len,@mft);
      printf ("%04d-%04d  Initialized attribute content size: ", $bpos, $bpos+$len-1);
      list_hex_digits ($bpos,$len,@mft);
      print " [", commify ($temp), " bytes]\n";

# If an attribute name is present, it will be here. $name_len is the lenth of the name
# in units of 2-bytes (i.e., $name_len = 4 means an 8-byte field)

      if ($name_len != 0)
        {
        $bpos += $len;
        $len = $name_len * 2;
        printf ("%04d-%04d  Name: ", $bpos, $bpos+$len-1);
        list_hex_digits ($bpos,$len,@mft);
        print " [";
        for ($k=0; $k<$name_len; $k++)
          {
          $temp = chr (little_endian ($bpos+$k*2,2,@mft));
          print $temp;
          }
        print "]\n";
        }

# Now, process the runlists. The runlists need to be sufficient to account for all of the
# file's VCNs. Also, the first run has an absolute cluster number but all subsequent runs
# list relative cluster number (from start of last run)

# AND... if the compression unit size ($cus) is not 0, only process first runlist entry

       if ($bpos+$len < $runlist_offset + $attrib_start)
         {
         list_bytes ($bpos+$len,$runlist_offset+$attrib_start-($bpos+$len),@mft);
         $bpos = $runlist_offset + $attrib_start;
         $len = 0;
         }

       $num_clus = $runlist_end - $runlist_start + 1;
       $relative = 0; $k = 0;
       while ($num_clus > 0)
         {

# If the first runlist value is not valid, then just jump out of runlist processing. Non-valid
# values have a 0x0 in the high- or low-order nibble.

         $temp = $mft [$bpos+$len];
         if ($temp % 16 == 0x0 or $temp < 0x10) { last; }

         $bpos += $len;
         $run_offset_len = int ($mft [$bpos] / 16);
         $run_length_len = $mft [$bpos] % 16;
         $len = $run_offset_len + $run_length_len + 1;
         $k++;
         printf ("%04d-%04d  Run list #%d: ", $bpos, $bpos+$len-1, $k);
         list_hex_digits ($bpos,$len,@mft);
         print "\n";
         $run_length = little_endian ($bpos+1,$run_length_len,@mft);
         print "           Run length ($run_length_len-byte value) is ", commify($run_length), " cluster(s)\n";
         if ($run_length > $num_clus)
           {
           print "           *** Generating invalid runlist results. Ignoring remainder of field. ***\n";
           last;
           }
         print "           Starting cluster ($run_offset_len-byte value) is #";
         if ($cus == 0)
             { $num_clus -= $run_length; }
           else
             { $num_clus = 0; }
         
# Use absolute cluster for first run and then relative clusters thereafter

         if ($relative == 0)
             {
             $run_offset = little_endian ($bpos+$run_length_len+1,$run_offset_len,@mft);
             print commify ($run_offset), "\n";
             $relative = 1;
             $cluster_num = $run_offset;
             }
           else
             {
             $run_offset = "";
             for ($j=$run_offset_len; $j>=1; $j--)
               {
               $temp = sprintf ("%02x", $mft[$bpos+$run_length_len+$j]);
               $run_offset = $run_offset.$temp;
               }
             $temp = hex_to_signed_integer ($run_offset,$run_offset_len);
             $cluster_num += $temp;
             print commify ($cluster_num), " (relative ";
             if ($temp<0)
                 { print "-", commify(abs($temp)), " clusters)\n"; }
               else
                 { print "+", commify($temp), " clusters)\n"; }
             }
         }

         $bpos += $len;

# Is there anything left in the attribute field?

         if ($bpos < $attrib_len + $attrib_start)
           {
           print "\nRemainder of Attribute field (possibly fill or 32-bit alignment...)\n";
           list_bytes ($bpos,$attrib_len+$attrib_start-$bpos,@mft);
           $bpos = $attrib_len + $attrib_start;
           }

      }

  }

if ($bpos < 1024)
  {
  print "\nRemainder of MFT contents...\n";
  list_bytes ($bpos,1024-$bpos,@mft);
  }

}

# ********************************************************
# parse_MFT_header
# ********************************************************

# Parse the MFT header

sub parse_MFT_header
{
my ($a, $attr_offset, $b, $fa_offset, $fa_size, $flag, $temp);

my %hdr_flags =
  (
  0x0000 => "Record is for a file (deleted)",
  0x0001 => "Record is a file",
  0x0002 => "Record is a directory (deleted)",
  0x0003 => "Record is a directory",
  0x0004 => "Don't know",
  0x0008 => "Don't know"
  );

print "********** MFT HEADER **********\n";

# Bytes 0-3 should be "FILE" or "BAAD"

$bpos = 0; $len = 4;
$temp = pack ("C4",@mft);
printf ("%04d-%04d  Signature: ", $bpos,$bpos+$len-1);
list_hex_digits ($bpos, $len, @mft);
print " [", $temp, "]\n";

if ($temp ne "FILE" && $temp ne "BAAD")
  {
  print "\n*** Invalid signature ***\n\n";
  exit;
  }

# Bytes 4-5 are the offset to fixup array. The fixup array begin right after
# byte 41 of the MFT header

$bpos = 4; $len = 2;
$fa_offset = little_endian ($bpos,$len,@mft);
printf ("%04d-%04d  Offset to fixup array: ", $bpos, $bpos+$len-1);
list_hex_digits ($bpos,$len,@mft);
print " [", commify ($fa_offset), "]\n";

# Bytes 6-7 are number of entries in fixup array

$bpos = 6; $len = 2;
$fa_size = little_endian ($bpos,$len,@mft);
printf ("%04d-%04d  Number of entries in fixup array: ", $bpos, $bpos+$len-1);
list_hex_digits ($bpos,$len,@mft);
print " [", commify ($fa_size), "]\n";

# Bytes 8-15 are $LogFile sequence number

$bpos = 8; $len = 8;
$temp = little_endian ($bpos,$len,@mft);
printf ("%04d-%04d  \$LogFile sequence number: ", $bpos, $bpos+$len-1);
list_hex_digits ($bpos,$len,@mft);
print " [", commify ($temp), "]\n";

# Bytes 16-17 are sequence value

$bpos = 16; $len = 2;
$temp = little_endian ($bpos,$len,@mft);
printf ("%04d-%04d  Sequence value: ", $bpos, $bpos+$len-1);
list_hex_digits ($bpos,$len,@mft);
print " [", commify ($temp), "]\n";

# Bytes 18-19 are link count

$bpos = 18; $len = 2;
$temp = little_endian ($bpos,$len,@mft);
printf ("%04d-%04d  Link count: ", $bpos, $bpos+$len-1);
list_hex_digits ($bpos,$len,@mft);
print " [", commify ($temp), "]\n";

# Bytes 20-21 are offset to first attribute. The MFT header, then, is
# $attr_offset bytes and the first attribute begins at this location.

$bpos = 20; $len = 2;
$attr_offset = little_endian ($bpos,$len,@mft);
printf ("%04d-%04d  Offset to first attribute: ", $bpos, $bpos+$len-1);
list_hex_digits ($bpos,$len,@mft);
print " [", commify ($attr_offset), "]\n";

# Bytes 22-23 are the flag bits (see table)

$bpos = 22; $len = 2;
$temp = little_endian ($bpos,$len,@mft);
printf ("%04d-%04d  Flags: ", $bpos, $bpos+$len-1);
list_hex_digits ($bpos,$len,@mft);
print " -- ";
if ($temp >= 0 && $temp <= 3)
    { print $hdr_flags{$temp}, "\n"; }
  else
    { print "undefined)\n"; }

# Bytes 24-27 are the actual number of bytes used in this record

$bpos = 24; $len = 4;
$temp = little_endian ($bpos,$len,@mft);
printf ("%04d-%04d  Logical size of record: ", $bpos, $bpos+$len-1);
list_hex_digits ($bpos,$len,@mft);
print " [", commify ($temp), " bytes]\n";

# Bytes 28-31 are size of MFT record

$bpos = 28; $len = 4;
$temp = little_endian ($bpos,$len,@mft);
printf ("%04d-%04d  Physical size of record: ", $bpos, $bpos+$len-1);
list_hex_digits ($bpos,$len,@mft);
print " [", commify ($temp), " bytes]\n";

# Bytes 32-39 are file reference to base record

$bpos = 32; $len = 8;
$temp = little_endian ($bpos,$len,@mft);
printf ("%04d-%04d  File reference to base record: ", $bpos, $bpos+$len-1);
list_hex_digits ($bpos,$len,@mft);
print " [", commify ($temp), "]\n";

# Bytes 40-41 are the identifier for the next attribute if one were to be added

$bpos = 40; $len = 2;
$temp = little_endian ($bpos,$len,@mft);
printf ("%04d-%04d  Id. of next attribute: ", $bpos, $bpos+$len-1);
list_hex_digits ($bpos,$len,@mft);
print " [", commify ($temp), "]\n";

# The fixup array starts at byte offset $fa_offset. The space between byte 42 and
# $fa_offset is fill, so just print those bytes.

$bpos = 42;
if ($bpos < $fa_offset)
  {

#  $len = $fa_offset - $bpos;
#  if ($len == 1)
#      { printf ("%04d       Fill: ", $bpos); }
#    else
#      { printf ("%04d-%04d  Fill: ", $bpos, $bpos+$len-1); }

  $len = 2;
  printf ("%04d-%04d  (Fill): ", $bpos, $bpos+$len-1);
  list_hex_digits ($bpos,$len,@mft);
  print "\n";

  $bpos += $len; $len = 4;
  $temp = little_endian ($bpos,$len,@mft);
  printf ("%04d-%04d  MFT record number: ", $bpos, $bpos+$len-1);
  list_hex_digits ($bpos,$len,@mft);
  print " [", commify ($temp), "]\n";
  }

# Two bytes starting at $fa_offset are the fixup array sequence number

$bpos += $len; $len = 2;
$temp = little_endian ($bpos,$len,@mft);
printf ("%04d-%04d  Fixup array sequence number: ", $bpos, $bpos+$len-1);
list_hex_digits ($bpos,$len,@mft);
print " [", commify ($temp), "]\n";

# The next $fa_size*2 bytes are the fixup array entries

$bpos += $len; $len = $fa_size*2;
printf ("%04d-%04d  %d fixup array entries: ", $bpos, $bpos+$len-1, $fa_size);
list_hex_digits ($bpos,$len,@mft);
print "\n";

# TEST. $bpos should now be set to the location of the first attribute. If not;
# there's a problem!

$bpos += $len;
if ($bpos > $attr_offset)
  {
  print "\n *** ERROR. End of MFT Header is beyond start of first attribute. ***\n\n";
  exit;
  }
}

# ********************************************************
# set_output
# ********************************************************

# Print output of the data bytes to show both hex and ASCII
# values.

sub set_output ($$@)
{
my ($start,$n,@array) = @_;
my ($j, $k, $linemax);

# Format for a maximum of 16 items per line

$linemax = 16;

printf ("%04d: ", $start);
for ($j=0; $j<$n; $j++)
  {
  printf (" %02X", $array[$start+$j]);
  }
if ($n < $linemax)
  {
  for ($j=1; $j<=$linemax-$n; $j++)
    { print "   "; };
  }

print "   ";
for ($j=0; $j<$n; $j++)
  {
  $k = $array[$start+$j];

#   Print a dot (.) for unprintable characters

  if ($k <= 0x1f || ($k >= 0x7f && $k <= 0x9f))
      { print "."; }
    else
      { print chr($k); }
  }
print "\n";

return;
}

